<?php

/**
 *  微信账号资源
 */
namespace App\Repositories\Business\Account;

use App\Libraries\pay\wxpay\PayApi;
use App\Libraries\Wx\WxApi;
use Exception;
use Illuminate\Support\Facades\DB;

class WechatRepository
{
    use \App\Helpers\ApiResponse;
    private $member_user;//普通用户表
    private $wechat_user;//微信用户表

    //初始化需要用到的表名称
    public function __construct()
    {
        $this->member_user      = config("db_table.member_user");
        $this->wechat_user      = config("db_table.wechat_user");
    }


    /**
     * 微信授权登陆
     * @param $code
     * @return mixed|null
     */
    public function login($code,$appid,$appSecret)
    {

        //取accessToken缓存信息
        $result     = $this->getCache($appid);

        //如果为空
        if(empty($result))
        {
            try{
                $result = WxApi::getInstance()->snsOauth2AccessToken($appid,$appSecret,$code);
            }
            catch(Exception $e)
            {
                $this->failed("获取accessToken失败:".$e->getMessage());
            }
        }


        //如果过期则刷新获取accessToken
        if(isset($result["expires_time"]) && time() > $result['expires_time'] && isset($result["refresh_token"]))
        {
            try{
                $result = WxApi::getInstance()->snsOauth2RefreshToken($appid, $result['refresh_token']);
            }
            catch(Exception $e)
            {
                $this->failed("刷新accessToken失败:".$e->getMessage());
            }
        }


        //缓存accessToken信息组
        if(isset($result["access_token"]) && isset($result['expires_in']))
        {
            $result['expires_time'] = time() + $result['expires_in'] - 100;
            unset($result['expires_in']);
            $this->setCache($appid,$result);
        }


        return $result;

        //通过openid 查询是否存在该微信用户
    }



    /**
     * 获取微信账号
     * @param $openid
     * @return mixed
     */
    protected function getWechatAccount($openid)
    {
        $where = [
            "openid"    => $openid,
        ];

        $user = DB::table($this->wechat_user)->where($where)->first();

        if(empty($user))
            $user->account_type = "wechat";

        return $user ? $user:null;
    }


    /**
     * @param $wechatId
     * @return mixed
     */
    protected  function getMainAccount($wechatId)
    {
        $memberRepos = new MemberRepository();
        $user = $memberRepos->getAccountByBindId("wechat",$wechatId);

        return $user;
    }



    /**
     * 创建微信账号
     */
    protected  function addAccount($result)
    {

        try {
            $row = WxApi::getInstance()->snsUserInfo($result["openid"], $result["access_token"]);
        }
        catch(Exception $e)
        {
            $this->failed("拉取用户信息失败:".$e->getMessage());
        }

        $params               = new \stdClass();
        $params->nick         = $row['nickname'];
        $params->header       = $row['headimgurl'];
        $params->sex          = $row['sex'] == 1 ? "man" :($row['sex'] == 2 ? 'woman':"unknown");
        $params->unionid      = $row['unionid'];
        $params->ip           = $_SERVER["REMOTE_ADDR"];
        $params->region       = $row['city'];
        $params->openid       = $row['openid'];
        $params->create_time  = date("Y-m-d H:i:s");


        $wechatId = DB::table($this->wechat_user)->insertGetId($params);

        $params->wechat_id    = $wechatId;
        $params->account_type = "wechat";
        $params->is_bind      = 0;

        return $wechatId ? $params:null;
    }

    /**
     * 微信浏览器获取支付参数
     * @param $openId
     * @param $orderNo
     * @param $money
     * @return bool|mixed
     */
    public function getPayParams($openId, $orderNo, $money, $title, $wxAppid, $mch_id, $mch_key, $uid)
    {
        $ip         = $_SERVER["REMOTE_ADDR"];

        //唤起第三方支付
        $params['openid']       = $openId;
        $params['out_trade_no'] = $orderNo;//测试环境id 可能与正式环境相冲突，订单会重复
        $params['total_fee']    = $money;//现价+邮费
        $params['body']         = $title;
        $params['ip']           = $ip;
        $params['notify_url']   = config("app.wxpay_callback_url");


        $wxpay = new PayApi($wxAppid,$params['notify_url'],$mch_id,$mch_key);
        $res   = $wxpay->order($params,$uid);

        return $res;
    }

    /**
     * 非微信浏览器支付
     * @param $orderNo
     * @param $money
     * @return bool|mixed
     */
    public function getOtherParams($orderNo, $money, $title, $wxAppid, $mch_id, $mch_key, $uid)
    {
        $ip                     = $_SERVER["REMOTE_ADDR"];
        //唤起第三方支付
        $params['out_trade_no'] = $orderNo;//测试环境id 可能与正式环境相冲突，订单会重复
        $params['total_fee']    = $money;//现价+邮费
        $params['ip']           = $ip;
        $params['notify_url']   = config("app.wxpay_callback_url");

        $params['body']         = $title;

        $wxpay = new PayApi($wxAppid,$params['notify_url'],$mch_id,$mch_key);
        $res   = $wxpay->otherOrder($params,$uid);

        return $res;
    }




    /**
     * 获取缓存
     * @param $appid
     * @return mixed
     */
    protected function getCache($appid)
    {
        $strKey     = "_accessToken_result";
        //$result     = Cache::get($appid.$strKey);
        $result     = session()->get($appid.$strKey);

        return $result;
    }


    /**
     * 设置缓存
     * @param $appid
     * @param $result
     * @return bool
     */
    protected function setCache($appid,$result)
    {
        $strKey     = "_accessToken_result";
        //$result     = Cache::forever($appid.$strKey,$result);

        $result     = session()->put($appid.$strKey,$result);

        return $result;
    }
}