<?php

namespace App\DataTables;

use App\Traits\DataTableTrait;

use App\Models\Announcement;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class AnnouncementDataTable extends DataTable
{
    use DataTableTrait;
    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        $query->orderBy( 'created_at', 'desc' );
        return datatables()
            ->eloquent($query)
            ->editColumn('author_id' , function ($announcement){
                return ($announcement->author_id != null && isset($announcement->author)) ? $announcement->author->display_name : '';
            })
            ->filterColumn('author_id',function($query,$keyword){
                $query->whereHas('providers',function ($q) use($keyword){
                    $q->where('display_name','like','%'.$keyword.'%');
                });
            })
            ->editColumn('status', function ($announcement) {
                $disabled = $announcement->trashed() ? 'disabled' : '';
                return '<div class="custom-control custom-switch custom-switch-text custom-switch-color custom-control-inline">
                    <div class="custom-switch-inner">
                        <input type="checkbox" class="custom-control-input bg-success change_status" data-type="announcement_status" ' . ($announcement->status ? "checked" : "") . '  ' . $disabled . ' value="' . $announcement->id . '" id="' . $announcement->id . '" data-id="' . $announcement->id . '">
                        <label class="custom-control-label" for="' . $announcement->id . '" data-on-label="" data-off-label=""></label>
                    </div>
                </div>';
            })
            ->addColumn('action', function ($announcement) {
                return view('announcement.action', compact('announcement'))->render();
            })
            ->addIndexColumn()
            ->rawColumns(['action', 'status', 'is_featured']);
    }

    /**
     * Get query source of dataTable.
     *
     * @param \App\Models\Announcement $model
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(Announcement $model)
    {
        if (auth()->user()->hasAnyRole(['admin'])) {
            $model = $model->withTrashed();
        }
        return $model->newQuery();
    }
    /**
     * Get columns.
     *
     * @return array
     */
    protected function getColumns()
    {
        return [
            Column::make('DT_RowIndex')
                ->searchable(false)
                ->title(__('messages.no'))
                ->orderable(false),
            Column::make('title'),
            Column::make('author_id')->title(__('messages.author')),
            Column::make('total_views'),
            Column::make('status'),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ];
    }

    // public function html()
    // {
    //     return $this->builder()
    //     ->columns($this->getColumns())
    //     // ->dom('lfrtip')
    //     ->language(['url' => '//cdn.datatables.net/plug-ins/9dcbecd42ad/i18n/Chinese.json']);
    // }

    /**
     * Get filename for export.
     *
     * @return string
     */
    protected function filename()
    {
        return 'announcement_' . date('YmdHis');
    }
}
