<?php

namespace App\Http\Controllers\API;

use App\Models\User;
use App\Models\OTP;
use Carbon\Carbon;
use Auth;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Services\SmsService;

class OTPController extends Controller
{

    private $expiration = 1; //設定有效時間3分鐘

    private $require_limit = 500; //最多嘗試幾次
		
	private $test_mode = false;
		
	private $code_length = 6; //驗證碼長度

    public function sms_test(){
        
        $success = false;
        $message = null;
        $balance = 0;
        $data = array();

        $phones = $_POST['dstaddr'];
        $smbody = $_POST['sendmessage'];

        $sms = app('sms');
        
        if( strpos( $phones,',' ) !== false ){
            $_phones = explode( ',', $phones );
            foreach( $_phones as $k => $phone ){
                $data[ $k+1 ] = [
                    'dstaddr'   => $phone,
                    'smbody'    => $smbody
                ];
            }
        }elseif( preg_match( '/^09[\d]{8}$/', $phones ) ){
            $data[] = [
                'dstaddr'   => $phones,
                'smbody'    => $smbody
            ];
        }else{
            $message = '輸入的手機格式不正確。';
        }

        if( empty( $message ) ) {
            try {
                $response = $sms->send($data);
                $message = iconv('big5', 'utf-8', $response);
                $response = preg_replace("/[\s]{2,}/","&",$response);
                parse_str($response, $arr);
                if( isset($arr['AccountPoint']) ){
                    $balance = $arr['AccountPoint'];
                }
                $success = true;
            } catch (Exception $e) {
                $message = $e->getMessage();
            }
        }

        return response()->json([
            'success' => $success,
            'message' => $message,
            'balance' => $balance
        ]);
    }

    public function detect_is_user_exists(Request $request){

        $data = $request->all();
        $status = 1;
        $msg = '不明錯誤';
        $expiration = $this->expiration * 60;
        if ($data['method'] == 'register') {
            return $this->user_register( $data );
        } elseif ($data['method'] == 'lost_password') {
            return $this->user_lost_password( $data );
        } elseif ( $data['method'] == 'change_password' ) {
            return $this->user_change_password( $data );
        }

        return response()->json(compact('status', 'msg', 'expiration'));
    }

    public function is_phone_number( $phone ){
        if( ! preg_match( '/^09[\d]{8}/', $phone ) ){
            return false;
        }else{
            return true;
        }
    }

    public function user_register( $data ){
        $user_login = $data['phone'];
        $status = $msg = '';
        $expiration = $this->expiration * 60;
        if( ! $this->is_phone_number( $user_login ) ){
            $status = 1;
            $msg = $this->get_error_message( 'no_phone' );
        }else{
            $user_id = $this->username_exists($user_login);
            if ($user_id) {
                //2021-09-07欲註冊帳號卻已存在
                $status = 1;
                $msg = $this->get_error_message('username_exists');

            } else {
                //如果check_user_value取不到IP或cookie就可能會發生錯誤，但不應該發生，所以不另外處理。
                $check_user_value = $this->get_user_check_value();
                $verify_code = $this->generate_verify_code($check_user_value, $user_login, $data['method']);
                if (is_numeric($verify_code)) {
                    $sms_result = $this->send_code_via_sms($user_login, $verify_code, 2);
                    
                    if( true !== $sms_result ){
                        OTP::where('phone', $user_login)->where('code', $verify_code)->delete();
                        $status = 1;
                        $msg = $sms_result;
                    }
                } else {
                    $status = 1;
                    $msg = $verify_code;
                }
            }
        }

        return response()->json( compact('status', 'msg', 'expiration') );
    }

    public function user_lost_password( $data ){
        $user_login = $data['phone'];
        $status = $msg = '';
        $expiration = $this->expiration * 60;
        if( ! $this->is_phone_number( $user_login ) ){
            $status = 1;
            $msg = $this->get_error_message( 'no_phone' );
        }else{
            $user_id = $this->username_exists($user_login);
            if ($user_id) {
                // $status = 1; //使用者存在
                //2021-09-06準備做簡訊認證碼
                $check_user_value = $this->get_user_check_value();
                $verify_code = $this->generate_verify_code($check_user_value, $user_login, $data['method']);
                if (is_numeric($verify_code)) {
                    $sms_result = $this->send_code_via_sms($user_login, $verify_code, 3);
                    if( true !== $sms_result ){
                        OTP::where('phone', $user_login)->where('code', $verify_code)->delete();
                        $status = 1;
                        $msg = $sms_result;
                    }
                } else {
                    $status = 1;
                    $msg = $verify_code;
                }
            } else {
                $status = 1; // 使用者不存在
                if ($this->is_phone_number( $user_login )) {
                    $msg = '請確認您欲驗證的方式是否正確';
                } else {
                    $msg = '您所輸入的資料尚未註冊';
                }
            }
        }
        return response()->json(compact('status', 'msg', 'expiration'));
    }

    public function user_change_password($data){
        $status = $msg = '';
        $expiration = $this->expiration * 60;
        if (Auth::check()) {
            $user_login = Auth::user()->phone;
            // $status = 1; //使用者存在
            //2021-09-06準備做簡訊認證碼
            $check_user_value = $this->get_user_check_value();
            $verify_code = $this->generate_verify_code($check_user_value, $user_login, $data['method']);
            if (is_numeric($verify_code)) {

                $sms_result = $this->send_code_via_sms($user_login, $verify_code, 3);

                if( true !== $sms_result ){
                    OTP::where('phone', $user_login)->where('code', $verify_code)->delete();
                    $status = 1;
                    $msg = $sms_result;
                }
            } else {
                $status = 1;
                $msg = $verify_code;
            }
        } else {
            $status = 1;
            $msg = '尚未登入無法更換密碼';
        }
        return response()->json(compact('status', 'msg', 'expiration'));
    }

    // 輸入驗證碼
    public function code_verifying(Request $request)
    {
        $status = $msg = $url = '';
        $data = $request->all();
        /* 如果是忘記密碼，則 transient suffix會是手機帳號
         * 如果是註冊，則transient suffix會是IP
         * */
        $code_transient = $this->get_verify_code($data['method'], $data['code']);
        if ($code_transient) {
            if (is_numeric($data['code']) && $data['code'] == $code_transient->code) {
                $code_transient->status = 1;
                $code_transient->save();
				if( $data['method'] == 'register' ){
					$status = 1; //驗證ok
				}else if( $data['method'] == 'lost_password' ){
					$status = 9; //驗證ok
				}else if( $data['method'] == 'change_password' ){
                    $status = 1; //驗證ok
                }
                //驗證碼驗證完成還不要移除，因為若此時清空了，但使用者還沒真的用手機去註冊，便可能可以重新再寄簡訊，應該讓記錄自動過期
                //$this->clear_gc_verify_transient();
            } else {
                $status = 2; //驗證失敗
                $msg = $this->get_error_message('verify_code_not_match');
            }
        } else {
            //若是false，則要求使用者重新索取驗證碼
            $status = 3;
            $msg = $this->get_error_message('verify_code_expired');
        }

        return response()->json(compact('status', 'msg', 'url'));
    }

    public function get_verify_code($method, $user_value = '')
    {
        $transient_suffix = $this->get_user_check_value();
        if ($transient_suffix) {
            $transient_name = $this->get_verify_code_transient_name($transient_suffix);
            $last_verofy_code = OTP::where('user_check', $transient_name)->where('content_type', $method)->where( 'created_at', '>=',  Carbon::now()->subMinute($this->expiration))->first();
            return $last_verofy_code;
        } else {
            return false;
        }
    }

    /**
     * @param value String,可為user_id或IP(點以底線代替)
     * @param Return number is success, otherwise return string.
     * */
    public function generate_verify_code($user_value, $user_login, $content_type)
    {
        $code_transient_name = $this->get_verify_code_transient_name($user_value);
        $code_transient = $this->get_verify_code_transient($code_transient_name);
        //2023-01-04如果有取得驗證碼，代表上一個還沒過期
        if( ! $code_transient ) {
            $verify_code = $this->create_verify_code();
            //檢查請求次數是否已達上限
             $times_transient = $this->get_verify_times_transient($code_transient_name, $content_type);
            if ($times_transient && $times_transient >= $this->require_limit) {
                return sprintf('請求驗證次數已達%d次的上限，請於24小時之後再試或透過右上角的會員中心登入。', $this->require_limit);
            } else {
                $this->set_verify_code_transient($code_transient_name, $verify_code, $user_login, $content_type);
            }
        }else{
            $verify_code = '請輸入先前取得的驗證碼，或稍候' . $this->expiration. '分鐘後再試。';
        }
        return $verify_code;
    }

    public function set_verify_code_transient($transient_suffix = '', $verify_code, $user_login, $content_type)
    {
        if ($transient_suffix) {
            OTP::create([
                'user_check' => $transient_suffix,
                'phone' => $user_login,
                'code' => $verify_code,
                'content_type' => $content_type
            ]);
        }
    }

    public function send_code_via_sms($phone, $code, $content_type = 0 )
    {
        if (!$this->test_mode) {
            //2021-09-06觸發三竹簡訊
            $sms = app('sms');
            if($sms) {
                
                $data[] = [
                    'DestName' => $phone,
                    'dstaddr' => $phone,
                    'smbody' => $this->get_verify_code_message($code, $content_type ),
                ];
                
                try {
                    
                    $response = $sms->send( $data );
                    if( $response ){
                        $results = parse_ini_string( $response );
                        if( isset( $results['Error'] ) && $results['Error'] ){
                            return $this->get_error_message( 'failed_sms' );
                        }else{
                            return true;
                        }
                    }else{
                        return $this->get_error_message( 'failed_sms' );
                    }
                }catch( Exception $e ){
                    return $e->getMessage();
                }
            }
        }
    }

    public function get_verify_code_message($code, $content_type )
    {
        $content = '的';
        $default_content = '家顧者%s驗證碼為%s';
        switch( $content_type ){
            case 2: //註冊會員
                $content = '註冊會員';
                break;
            case 3: //忘記密碼
                $content = '忘記密碼';
                break;
            case 4: //更換密碼
                $content = '更換密碼';
                break;
        }
        return sprintf( $default_content, $content, $code );
    }

    public function create_verify_code()
    {
        $code = '';
        $length = 6;

        for ($i = 0; $i < $length; $i++) {
            $code .= rand(0, 9);
        }

        return $code;
    }

    public function get_verify_times_transient($transient_suffix = '', $content_type){
        $times = OTP::where('user_check', $transient_suffix)->where('content_type', $content_type)->where('created_at', '>=', Carbon::today() )->count();
        return $times;
    }

    public function get_verify_code_transient($transient_suffix = '')
    {
        if ($transient_suffix) {
            $last_verofy_code = OTP::where('user_check', $transient_suffix)->where( 'created_at', '>=',  Carbon::now()->subMinute($this->expiration))->first();
            
            if( $last_verofy_code ){
                return true;
            }else{
                return false;
            }
        }
        return false;
    }

    public function get_verify_code_transient_name($transient_suffix = '')
    {
        return $transient_suffix && is_scalar($transient_suffix) ? 'gc_verify_code_' . $transient_suffix : false;
    }

    public function username_exists($phone){
        $user = User::where('contact_number', $phone)->withTrashed()->first();
        if( $user ){
            return $user->id;
        }else{
            return false;
        }
    }

    public function get_user_check_value()
    {
        $check_user_value = '';

        //2021-09-07帳號不存在，準備驗證碼
        $check_user_value = $this->get_user_ip();

        //私有網域可能會取得空值，所以就改用uniqid並存放在cookie
        if (empty($check_user_value)) {
            if (isset($_COOKIE['gc_check_user_value'])) {
                $check_user_value = $_COOKIE['gc_check_user_value'];
            } else {
                $check_user_value = uniqid();
                setcookie('gc_check_user_value', $check_user_value, 0);
            }
        }

        return $check_user_value;
    }

    public function get_user_ip()
    {
        foreach (array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR') as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (array_map('trim', explode(',', $_SERVER[$key])) as $ip) {
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
    }

    public function get_error_message($error_code = '')
    {
        $error_codes = [
            'nonce_verify_failed' => '驗證失敗，請稍候再試',
            'verify_code_expired' => '驗證碼過期，請重新取得驗證碼',
            'verify_code_not_match' => '驗證碼有誤，請重新輸入',
            'username_exists' => '該手機已被使用或帳號已存在，請直接登入',
            'no_phone' => '請輸入正確的手機號碼',
            'failed_sms' => '無法寄送簡訊，請聯繫客服',
        ];

        if ($error_code && isset($error_codes[$error_code])) {
            return $error_codes[$error_code];
        } else {
            return '未知的錯誤。';
        }
    }

    
}
