import { __extends } from "tslib";
import './polyfills';
import { scrollbarMap, Scrollbar, } from './scrollbar';
import { addPlugins, ScrollbarPlugin, } from './plugin';
import { attachStyle, detachStyle, } from './style';
export { ScrollbarPlugin };
/**
 * cast `I.Scrollbar` to `Scrollbar` to avoid error
 *
 * `I.Scrollbar` is not assignable to `Scrollbar`:
 *     "privateProp" is missing in `I.Scrollbar`
 *
 * @see https://github.com/Microsoft/TypeScript/issues/2672
 */
var SmoothScrollbar = /** @class */ (function (_super) {
    __extends(SmoothScrollbar, _super);
    function SmoothScrollbar() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    /**
     * Initializes a scrollbar on the given element.
     *
     * @param elem The DOM element that you want to initialize scrollbar to
     * @param [options] Initial options
     */
    SmoothScrollbar.init = function (elem, options) {
        if (!elem || elem.nodeType !== 1) {
            throw new TypeError("expect element to be DOM Element, but got " + elem);
        }
        // attach stylesheet
        attachStyle();
        if (scrollbarMap.has(elem)) {
            return scrollbarMap.get(elem);
        }
        return new Scrollbar(elem, options);
    };
    /**
     * Automatically init scrollbar on all elements base on the selector `[data-scrollbar]`
     *
     * @param options Initial options
     */
    SmoothScrollbar.initAll = function (options) {
        return Array.from(document.querySelectorAll('[data-scrollbar]'), function (elem) {
            return SmoothScrollbar.init(elem, options);
        });
    };
    /**
     * Check if there is a scrollbar on given element
     *
     * @param elem The DOM element that you want to check
     */
    SmoothScrollbar.has = function (elem) {
        return scrollbarMap.has(elem);
    };
    /**
     * Gets scrollbar on the given element.
     * If no scrollbar instance exsits, returns `undefined`
     *
     * @param elem The DOM element that you want to check.
     */
    SmoothScrollbar.get = function (elem) {
        return scrollbarMap.get(elem);
    };
    /**
     * Returns an array that contains all scrollbar instances
     */
    SmoothScrollbar.getAll = function () {
        return Array.from(scrollbarMap.values());
    };
    /**
     * Removes scrollbar on the given element
     */
    SmoothScrollbar.destroy = function (elem) {
        var scrollbar = scrollbarMap.get(elem);
        if (scrollbar) {
            scrollbar.destroy();
        }
    };
    /**
     * Removes all scrollbar instances from current document
     */
    SmoothScrollbar.destroyAll = function () {
        scrollbarMap.forEach(function (scrollbar) {
            scrollbar.destroy();
        });
    };
    /**
     * Attaches plugins to scrollbars
     *
     * @param ...Plugins Scrollbar plugin classes
     */
    SmoothScrollbar.use = function () {
        var Plugins = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            Plugins[_i] = arguments[_i];
        }
        return addPlugins.apply(void 0, Plugins);
    };
    /**
     * Attaches default style sheets to current document.
     * You don't need to call this method manually unless
     * you removed the default styles via `Scrollbar.detachStyle()`
     */
    SmoothScrollbar.attachStyle = function () {
        return attachStyle();
    };
    /**
     * Removes default styles from current document.
     * Use this method when you want to use your own css for scrollbars.
     */
    SmoothScrollbar.detachStyle = function () {
        return detachStyle();
    };
    SmoothScrollbar.version = "8.7.5";
    SmoothScrollbar.ScrollbarPlugin = ScrollbarPlugin;
    return SmoothScrollbar;
}(Scrollbar));
export default SmoothScrollbar;
//# sourceMappingURL=index.js.map